//*************************************************************************************************
//
//	Description:
//		carbon.fx - Special-purpose shader for carbon-fiber materials. Supports anisotropic
//								reflections to simulate woven carbon fiber.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     07/17/2006  0.1           Created
//		TNettleship     07/24/2007  0.2           Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  0.3						Removed texture bias.
//		TNettleship			10/23/2007	0.4						Converted to on-load rendermode behaviour binding.
//		TNettleship			12/03/2007	0.5						Fixed a bug with clearcoat fresnel.
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define SUNSPOT_SPECULAR_POWER 8192.0f
#define SUNSPOT_BRIGHTNESS_MULT 4.0f

#define MAX_ENVMAP_BIAS	6.0f


// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define BONNETCAM_ENVMAP
#define ENV_MAPPING
#define NORMAL_MAPPING
#define USE_VINYLS
#define ALLOW_VINYLS
#endif

// Uncomment to use experimental higher quality envmapping approach
#define EXPERIMENTAL_ENVMAPPING



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

#if defined( _3DSMAX_ )
float4x4 worldI : WorldInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
float4x4 envTransformI : EnvInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#elif defined( ENV_MAPPING )

#if !defined( BONNETCAM_ENVMAP )
float4x4 envTransformI : EnvInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
float4x4 view : View
<
	bool appEdit = false;
>;
#endif

#endif



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// UV channel used for vinyls, uses mapchannel 3 to match bodywork.fx.
// Also, is exported to texcoord2, again, to match bodywork.fx.
int texcoord4 : Texcoord
<
	string UIWidget = "None";
	int MapChannel = 3;
	int Texcoord = 3;
	int RuntimeTexcoord = 2;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
	string UIName = "Spec Tex {UV1}";
	bool appEdit = true;
>;

float fresnelFactor
<
	string UIName = "Clearcoat Fresnel";
	float UIMin = -10.0f;
	float UIMax = 10.0f;
	bool appEdit = true;
> = 0.666f;

texture anisotropyTexture : TEXTURE						// Anisotropic vector colour in RB
<
	string UIName = "Aniso Tex {UV1}";
	bool appEdit = true;
>;

float alphaX
<
	string UIName = "U spec sharpness";
	bool appEdit = true;
> = 0.2f;

float alphaY
<
	string UIName = "V spec sharpness";
	bool appEdit = true;
> = 0.2f;

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
DEPENDENT_BOOL_PARAM( specialNormalMap, "Special normal map format?", useNormalMap )
#endif

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )

 #if !defined( BONNETCAM_ENVMAP )

DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )

#else

texture motionBlurTexture : motionBlurTexture					// Bonnetcam environment map uses the motion blur texture
<
	bool appEdit = false;
>;

#endif

#endif

float minSpecPower
<
	string UIName = "CC Min Spec";
	float UIMin = 1.0f;
	float UIMax = 65536.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "CC Max Spec";
	float UIMin = 1.0f;
	float UIMax = 65536.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 1.0f, 1.0f )
#endif


// Vinyls (custom livery).
//

SPECIALISATION_PARAM( useVinyls, "Use vinyls?", "USE_VINYLS" )
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
DEPENDENT_TEXTURE_PARAM( vinylsTexture, "Vinyls Tex {UV3}", useVinyls )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vinylsSpecularFactor, "Specular Factor of Vinyls", useVinyls, 0.0f, 16.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vinylsWardAlpha, "Roughness of Vinyls", useVinyls, 0.0f, 2.0f, 0.2f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vinylsEMapFactor, "EMap Factor of Vinyls", useVinyls, 0.0f, 16.0f, 0.7f )
#endif
DECLARE_DEPENDENT_VERTEX_STREAM( uv2VinylsDependency, texCoord2, TEXCOORD3, useVinyls )

SPECIALISATION_PARAM( allowVinyls, "Allow vinyls?", "ALLOW_VINYLS" )
DECLARE_DEPENDENT_VERTEX_STREAM( uv2VinylsDependency2, texCoord2, TEXCOORD3, allowVinyls )


// Simplify shadowing code to access only the first (the nearest) shadowmap.
SPECIALISATION_PARAM( forceFistShadowmap, "Force the first shadowmap?", "FORCE_FIRST_SHADOWMAP" )


//
// Lighting
//

// Tell the lighting backend we want anisotropic specular calcs
// (N.B. This feels messy. Find a better way to do it.)
#define USE_WARD_ANISOTROPIC_SPECULAR

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };

SPECIALISATION_PARAM( bonnetCamEnvMap, "Bonnetcam Envmap?", "BONNETCAM_ENVMAP" )	// TRUE if special-case bonnetcam envmap should be used


#if defined( _3DSMAX_ )
#define GetLightDirection( _input ) _input.lightVector
#define GetLightColour( _input ) lightColour
#elif 		defined( _LIGHTING_D_ )		|| 	defined( _LIGHTING_DD_ )	|| 	defined( _LIGHTING_DDD_ ) || defined( _LIGHTING_DP_ )	||	defined( _LIGHTING_DS_ )	|| 	defined( _LIGHTING_DPS_ ) || defined( _LIGHTING_DPP_ )	|| 	defined( _LIGHTING_DSS_ ) || 	defined( _LIGHTING_DDP_ ) || defined( _LIGHTING_DDS_ ) 
#define GetLightDirection( _input ) lightDirection0
#define GetLightColour( _input ) lightColour0
#else
#define GetLightDirection( _input ) float3(0.0f,1.0f,0.0f)
#define GetLightColour( _input ) float4(1.0f, 1.0f, 1.0f, 1.0f)
#endif



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
<
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D anisotropyMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="anisotropyTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < anisotropyTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )

#if !defined( BONNETCAM_ENVMAP )

samplerCUBE environmentMap : SAMPLER 
<
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#else

sampler2D motionBlurMap : SAMPLER 
< 
#if defined(_PS3_)
	SET_SRGB_TEXTURE
#else
	SET_LINEAR_TEXTURE
#endif
	bool appEdit = false; 
	string SamplerTexture="motionBlurTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
	string texformat="COMPRESSED_RGBA_S3TC_DXT1";
> 
= sampler_state
{
	Texture = < motionBlurTexture >;
#if defined(SET_FX_SAMPLER_STATES)
#if defined(_PS3_)
	FX_SAMPLERSTATE_SRGB_TEXTURE
#else
	FX_SAMPLERSTATE_LINEAR_TEXTURE
#endif
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#endif

#endif


#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
sampler2D vinylsMap : SAMPLER
<
	SET_SRGB_TEXTURE
	bool appEdit = false;
	string SamplerTexture = "vinylsTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < vinylsTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif	// _3DSMAX_ || USE_VINYLS



//
// Functions
//


//
// This seems to work almost as well as the full-on "complicated fresnel"
//
float calculated_fresnel(float3 _eye, float3 _normal, float rindexRatio)
{
	// Note: compute R0 on the CPU and provide as a
	// constant; it is more efficient than computing R0 in
	// the vertex shader. R0 is:
	// float const R0 = pow(1.0-refractionIndexRatio, 2.0)
	// / pow(1.0+refractionIndexRatio, 2.0);
	// light and normal are assumed to be normalized
	float R0 = pow(1.0f - rindexRatio, 2.0f) / pow ( 1.0f + rindexRatio, 2.0f );

	return R0 + (1.0f-R0) * pow(1.0f - saturate( dot(-_eye, _normal) ), 5.0f);
}


//-----------------------------------------------------------------------
//## Get2DUVCoordinates()
//
//  Description:
//    Given a pair of UVs, convert them into normal form - it involves flipping V in Max.
//
float2 Get2DUVCoordinates( float2 uv )
{
#if defined( _3DSMAX_ )
	return float2( uv.x, 1.0f - uv.y );
#else
	return uv;
#endif	// ! _3DSMAX_
}


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
	float2 texCoord2 : TEXCOORD3;													// UV channel for vinyls.
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#if defined( USE_VINYLS )
	float2 texCoord2 : TEXCOORD2;													// UV channel for vinyls.
#endif	// USE_VINYLS
#endif	// ! _3DSMAX_
	float3 normal   : NORMAL;															// Object space normal
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
};



// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD5;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float4 normal_and_fresnel	: TEXCOORD1;								// Normal vector (world space) plus fresnel factor
	float4 eye				: TEXCOORD2;												// Eye vector (world space)
	float3 tangent		: TEXCOORD3;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD4;												// Binormal vector (world space)
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
	float2 texCoord2	: TEXCOORD6;												// UV coord for vinyls.
	DECLARE_LIGHTING_INTERPOLATORS_VS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_VS( 6 )
#endif
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT CarbonVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	DEPENDENT_CODE_START( useVinyls )
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
		_output.texCoord2 = _input.texCoord2;
#endif
	DEPENDENT_CODE_END( useVinyls )

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal_and_fresnel.xyz   = normal;
	_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
	_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;

	_output.normal_and_fresnel.w = calculated_fresnel( - normalize( worldEyeVec ), normal, fresnelFactor );

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour *= globalColourMultiplier;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour			: TEXCOORD5;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float4 normal_and_fresnel			: TEXCOORD1;						// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2;												// Eye vector (world space)
	float3 tangent		: TEXCOORD3;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD4;												// Binormal vector (world space)
	float2 texCoord2	: TEXCOORD6;												// UV coord for vinyls.

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD5;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float4 normal_and_fresnel			: TEXCOORD1_centroid;		// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)
	float3 tangent		: TEXCOORD3_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD4_centroid;								// Binormal vector (world space)
#if defined( USE_VINYLS )
	float2 texCoord2	: TEXCOORD6;												// UV coord for vinyls.
	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_PS( 6 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT CarbonFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	//
	// Read textures.
	//

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	float4 specularTexColour = tex2D( specularMap, _input.texCoord );
	float4 anisoMapColour = tex2D( anisotropyMap, _input.texCoord );

#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
	float4 vinylsColour = tex2D( vinylsMap, Get2DUVCoordinates( _input.texCoord2.xy ) );
#endif

	//
	// Block (USE_VINYLS):
	//   Prepare various vinyl-related factors that will get used in blocks below.
	//

	// "Patch" missing parameters when certain features are not turned on.
#if ! defined( _3DSMAX_ ) && ! defined( ENV_MAPPING )
	float globalEMapFactor = 1.0f;
#endif

	float emapFactor = globalEMapFactor;					// On pure carbon, use conventional material property to control the strength of reflections.
	float specularFactor = globalSpecularFactor;	// On pure carbon, specular strength is controlled by carbon material properties.
	float alphaXValue = alphaX;
	float alphaYValue = alphaY;
	float sunspotSpecularFactor = SUNSPOT_BRIGHTNESS_MULT;

	DEPENDENT_CODE_START( useVinyls )
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )

		// Change the intensity of specular and reflections / envmapping where covered by vinyls to the one provided by 'vinyl properties'.
		emapFactor = lerp( globalEMapFactor, vinylsEMapFactor, vinylsColour.a );
		specularFactor = lerp( globalSpecularFactor, vinylsSpecularFactor, vinylsColour.a );

		diffuseTexColour.rgb = lerp( diffuseTexColour.rgb, vinylsColour.rgb, vinylsColour.a );
		specularTexColour = lerp( specularTexColour, float4( 1.0f, 1.0f, 1.0f, 1.0f ), vinylsColour.a );

		alphaXValue = lerp( alphaXValue, vinylsWardAlpha, vinylsColour.a );
		alphaYValue = lerp( alphaYValue, vinylsWardAlpha, vinylsColour.a );

#endif
	DEPENDENT_CODE_END( useVinyls )

	//
  // Normalise interpolated vectors
	//

	float3 TSnormal = normalize( _input.normal_and_fresnel.xyz );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;
	float3 tangent = normalize( _input.tangent );
	float3 binormal = normalize( _input.binormal );

	//
	// Calculate the world coords anisotropic coordinate axes for this pixel
	//

	float3 anisoVector = ( anisoMapColour.xyz * 2.0f ) - 1.0f;
	anisoVector.y = 0.0f;
	anisoVector = normalize( anisoVector );

	float3 perturbedUAxis = ( anisoVector.x * binormal ) + ( anisoVector.z * tangent );
	float3 perturbedVAxis = ( -anisoVector.z * binormal ) + ( anisoVector.x * tangent );

	//
	// Block (NORMAL_MAPPING):
	//   Do tangent space normal mapping if required.
	//
	//   Sets the 'normal' variable.
	//

	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord );

		float3 normalFromMap;
		if ( specialNormalMap )
		{
			normalFromMap = normalize( ( normalMapColour.agb * 2.0f ) - 1.0f );
		}
		else
		{
			normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );
		}

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal and constant specular strength
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	//
	// Block:
	//   Ambient lighting.
	//

#if defined( PSEUDO_GI ) && !defined( BONNETCAM_ENVMAP )

	// If environment mapping is switched on
	float4 accumulatedColour;
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate ambient-lit base colour (looking up the envmap for the ambient lighting)
		float4 envmapambient = texCUBElod( environmentMap, float4( normal.xyz, 4 ) );
		accumulatedColour = diffuseTexColour * envmapambient;
#endif
	DEPENDENT_CODE_ELSE( useEnvironmentMap )
#if defined( _3DSMAX_ ) || !defined( ENV_MAPPING )
		// Calculate ambient-lit base colour from interpolated lighting
		accumulatedColour = diffuseTexColour * _input.colour;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

#else		// ! PSEUDO_GI
	float4 accumulatedColour = diffuseTexColour * _input.colour;
#endif	// ! PSEUDO_GI

	//
	// Block (ENV_MAPPING):
	//   Environment mapping (if switched on).
	//

	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )


//
// Standard envmapping approach
//
	float4 environmentTexColour;

#if !defined( BONNETCAM_ENVMAP )

	// Calculate the envmap space reflection vector
#if defined( _3DSMAX_ )
		// Max uses a map aligned to the object, always
		float3 envMapSpaceNormal = mul( float4( normal, 0.0f ), worldI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), worldI ).xyz;
#else
		// The engine uses a map aligned to its own space, which may be object aligned, or to a parent object
		float3 envMapSpaceNormal = mul( float4( normal, 0.0f ), envTransformI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), envTransformI ).xyz;
#endif
	float3 reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

#if defined( EXPERIMENTAL_ENVMAPPING )
	// Intersect the reflection vector with a 4m sphere centred on the car
	float3 envMapSpacePos = mul( float4( worldPos, 1.0f ), envTransformI ).xyz;
	float3 rayOrigin = envMapSpacePos;
	float3 rayDirection = reflectionVector;
	float B = dot( rayDirection, rayOrigin ) * 2.0f;
	float C = dot( rayOrigin, rayOrigin ) - ( 4.0f * 4.0f );
	float t0 = -B - sqrt( B*B - 4.0f * C ) / 2.0f;
	float t1 = -B + sqrt( B*B - 4.0f * C ) / 2.0f;
	float3 sphereIntersection = rayOrigin + ( rayDirection * max( t0, t1 ) );

	// Use the intersection point to look up the cubemap, not the reflection vector
	reflectionVector = normalize( sphereIntersection );
#endif

	// Fetch the environment map colour using the reflection vector
#ifdef _3DSMAX_
	environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
	environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

//
// Bonnetcam envmapping approach
//
#else		// !defined( BONNETCAM_ENVMAP )

	// Calculate the view space reflection vector
	float3 viewSpaceNormal = mul( float4( normal, 0.0f ), view ).xyz;
	float3 viewSpaceEye = mul( float4( -eye, 0.0f ), view ).xyz;
	float3 reflectionVector = reflect( viewSpaceEye, viewSpaceNormal );

	environmentTexColour = tex2Dbias( motionBlurMap, float4( 1.0f - ( ( -reflectionVector.x * 0.5f ) + 0.5f ), 1.0f - ( ( reflectionVector.y * 0.5f ) + 0.5f ), 0.0f, 1.4f ) );

#endif

	// Calculate envmap colour (back in tangent space is fine) and add to diffuse
	accumulatedColour += environmentTexColour * emapFactor * _input.normal_and_fresnel.w;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	//
	// Perform lighting
	//
	float globalSpecularFactorValue = specularFactor;
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	// Apply sunspot specular
#if defined( _3DSMAX_ )
	float PSL_attenuation = 1.0f;
#endif
	float3 tomtest_L = -GetLightDirection(_input);
	accumulatedColour += GetLightColour(_input) * specularTexColour * sunspotSpecularFactor * CalculateSpecularLighting( normal, tomtest_L, eye, SUNSPOT_SPECULAR_POWER ) * PSL_attenuation;

	accumulatedColour.w = _input.colour.w;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}




//
// Low Detail Shaders
//

struct VSOUTPUT_LD
{
	float4 position		: POSITION;												// View-coords position
	float4 colour			: TEXCOORD1;											// Vertex colour
	float2 texCoord		: TEXCOORD0;											// UV coords for texture channel 0
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
	float2 texCoord2	: TEXCOORD2;											// UV coord for vinyls.
#endif
};


VSOUTPUT_LD CarbonLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;

	DEPENDENT_CODE_START( useVinyls )
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
		_output.texCoord2 = _input.texCoord2;
#endif
	DEPENDENT_CODE_END( useVinyls )

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// get normal in world space and do lighting
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT CarbonLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
	float4 vinylsColour = tex2D( vinylsMap, Get2DUVCoordinates( _input.texCoord2.xy ) );
#endif

	DEPENDENT_CODE_START( useVinyls )
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
		diffuseTexColour.rgb = lerp( diffuseTexColour.rgb, vinylsColour.rgb, vinylsColour.a );
#endif
	DEPENDENT_CODE_END( useVinyls )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;
	_output.Colour = CalculateLowDetailOutputPixel ( accumulatedColour );

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Carbon
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Carbon";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Carbon_LowDetail";
	int    lowDetailDeferredID = 0;
	bool   usefullprecision = true;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx CarbonVertexShader();
		PixelShader = compile sce_fp_rsx CarbonFragmentShader();
#else
		VertexShader = compile vs_3_0 CarbonVertexShader();
		PixelShader = compile ps_3_0 CarbonFragmentShader();
#endif
	}
}


technique Carbon_LowDetail
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Carbon_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx CarbonLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx CarbonLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 CarbonLowDetailVertexShader();
		PixelShader = compile ps_3_0 CarbonLowDetailFragmentShader();
#endif
	}
}
